#ifndef XPTR_H_INCLUDED
#define XPTR_H_INCLUDED

#include <stdbool.h>

/**
 * \brief   Maximum number of xptrs stored in the xptr store.
 */
#define XPTR_STORE_SIZE 50

/**
 * \brief   Error indicating that the xptr store is full.
 */
#define XPTR_ERROR_STORE_FULL (-1)

/**
 * \brief   The xptr pointer type.
 */
typedef const void * xptr;

/**
 * \brief   Entry in the array used as xptr store.
 */
typedef struct
{
    xptr target;    /**< \brief   Target of the xptr. */
    bool used;      /**< \brief   Indicates if the entry is currently used. */
}
xptr_store_entry;

/**
 * \brief   Creates an xptr for a certain pointer.  Multiple xptrs can be
 *          created for a certain pointer.
 *
 * \param p[in]     Pointer to create an xptr for.
 *
 * \return  Returns the created xptr, `XPTR_ERROR_STORE_FULL` if the xptr could
 *          not be created because the store is full.
 */
xptr *xptr_create(const void *const p);

/**
 * \brief   Makes all xptrs pointing to a supplied pointer point to `NULL`.
 *          Should be called when the memory pointed to by the pointer is freed.
 *
 * \param p[in]     Pointer targeted by xptrs which should be set to `NULL`.
 */
void xptr_null(const void *const p);

/**
 * \brief   Frees a specific xptr that is not used any more.  Does not affect
 *          other xptrs pointing to the same pointer.
 *
 * \param xp[in]    The xptr to be freed.
 */
void xptr_free(const xptr *const xp);

#endif // XPTR_H_INCLUDED
